﻿Namespace Common
    ''' <summary>
    ''' NavigationHelper 协助在页面间进行导航。 它提供一些命令，用于
    ''' 前后导航以及注册用于前进和后退的标准鼠标和键盘
    ''' 导航请求快捷方式和 Windows Phone 中的
    ''' 硬件“后退”按钮。此外，它集成了 SuspensionManger 以在页面之间导航时处理
    ''' 进程生存期管理和状态管理。
    ''' </summary>
    ''' <remarks>
    ''' <example>
    ''' 若要利用 NavigationHelper，请执行下面两步或
    ''' 以 BasicPage 或除 BlankPage 以外的任何其他页项模板开始。
    ''' 
    ''' 1) 在某个位置创建 NaivgationHelper 的实例，例如在以下位置
    '''     页面的构造函数中)，并注册 LoadState 和
    '''     SaveState 事件的回调。
    ''' <code>
    '''     Public NotInheritable Class MyPage
    '''         Inherits Page
    ''' 
    '''         Public Sub New()
    '''             InitializeComponent();
    '''             Me._navigationHelper = New Common.NavigationHelper(Me)
    '''             AddHandler Me._navigationHelper.LoadState, AddressOf NavigationHelper_LoadState
    '''             AddHandler Me._navigationHelper.SaveState, AddressOf NavigationHelper_SaveState
    '''         End Sub
    '''     
    '''     Private Sub NavigationHelper_LoadState(sender As Object, e As Common.LoadStateEventArgs)
    '''     End Sub
    ''' 
    '''     Private Sub NavigationHelper_SaveState(sender As Object, e As Common.SaveStateEventArgs)
    '''     End Sub
    ''' </code>
    ''' 
    ''' 2) 在以下情况下注册页面以调入 NavigationHelper: 该页面
    '''      通过重写 <see cref="Windows.UI.Xaml.Controls.Page.OnNavigatedTo"/> 
    '''     和 <see cref="Windows.UI.Xaml.Controls.Page.OnNavigatedFrom"/> 事件以参与导航。
    ''' <code>
    '''     Protected Overrides Sub OnNavigatedTo(e As NavigationEventArgs)
    '''         _navigationHelper.OnNavigatedTo(e)
    '''     
    '''     Protected Overrides Sub OnNavigatedFrom(e As NavigationEventArgs)
    '''         _navigationHelper.OnNavigatedFrom(e)
    ''' </code>
    ''' </example>
    ''' </remarks>
    Public Class NavigationHelper
        Inherits DependencyObject
        Private Property Page() As Page
        Private ReadOnly Property Frame() As Frame
            Get
                Return Me.Page.Frame
            End Get
        End Property

        Public Sub New(page As Page)
            Me.Page = page

            ' 当此页是可视化树的一部分时，进行两个更改: 
            ' 1) 将应用程序视图状态映射到页的可视状态
            ' 2) 处理用于在 Windows 中向前和向后移动的
            AddHandler Me.Page.Loaded,
                Sub(sender, e)
#If WINDOWS_PHONE_APP Then
                    AddHandler HardwareButtons.BackPressed, AddressOf HardwareButtons_BackPressed
#Else
                    ' 仅当占用整个窗口时，键盘和鼠标导航才适用
                    If Me.Page.ActualHeight = Window.Current.Bounds.Height AndAlso Me.Page.ActualWidth = Window.Current.Bounds.Width Then
                        ' 直接侦听窗口，因此无需焦点
                        AddHandler Window.Current.CoreWindow.Dispatcher.AcceleratorKeyActivated, AddressOf CoreDispatcher_AcceleratorKeyActivated
                        AddHandler Window.Current.CoreWindow.PointerPressed, AddressOf Me.CoreWindow_PointerPressed
                    End If
#End If
                End Sub

            ' 当页不再可见时，撤消相同更改
            AddHandler Me.Page.Unloaded,
                Sub(sender, e)
#If WINDOWS_PHONE_APP Then
                    RemoveHandler HardwareButtons.BackPressed, AddressOf HardwareButtons_BackPressed
#Else
                    RemoveHandler Window.Current.CoreWindow.Dispatcher.AcceleratorKeyActivated, AddressOf CoreDispatcher_AcceleratorKeyActivated
                    RemoveHandler Window.Current.CoreWindow.PointerPressed, AddressOf Me.CoreWindow_PointerPressed
#End If
                End Sub
        End Sub

#Region "导航支持"

        Private _goBackCommand As RelayCommand
        Public Property GoBackCommand() As RelayCommand
            Get
                If _goBackCommand Is Nothing Then
                    _goBackCommand = New RelayCommand(AddressOf Me.GoBack, AddressOf Me.CanGoBack)
                End If
                Return _goBackCommand
            End Get
            Set(value As RelayCommand)
                _goBackCommand = value
            End Set
        End Property

        Private _goForwardCommand As RelayCommand
        Public ReadOnly Property GoForwardCommand() As RelayCommand
            Get
                If _goForwardCommand Is Nothing Then
                    _goForwardCommand = New RelayCommand(AddressOf Me.GoForward, AddressOf Me.CanGoForward)
                End If
                Return _goForwardCommand
            End Get
        End Property

        Public Overridable Function CanGoBack() As Boolean
            Return Me.Frame IsNot Nothing AndAlso Me.Frame.CanGoBack
        End Function
        Public Overridable Function CanGoForward() As Boolean
            Return Me.Frame IsNot Nothing AndAlso Me.Frame.CanGoForward
        End Function

        Public Overridable Sub GoBack()
            If Me.Frame IsNot Nothing AndAlso Me.Frame.CanGoBack Then
                Me.Frame.GoBack()
            End If
        End Sub
        Public Overridable Sub GoForward()
            If Me.Frame IsNot Nothing AndAlso Me.Frame.CanGoForward Then
                Me.Frame.GoForward()
            End If
        End Sub

#If WINDOWS_PHONE_APP Then
        '''<summary>
        ''' 处理后退按钮按下事件并在根框架的历史记录中导航。
        ''' </summary>
        Private Sub HardwareButtons_BackPressed(sender As Object, e As BackPressedEventArgs)
            If Me.GoBackCommand.CanExecute(Nothing) Then
                e.Handled = True
                Me.GoBackCommand.Execute(Nothing)
            End If
        End Sub
#Else
        ''' <summary>
        ''' 当此页处于活动状态并占用整个窗口时，在每次
        ''' 击键(包括系统键，如 Alt 组合键)时调用。    用于检测页之间的键盘
        ''' 导航(即使在页本身没有焦点时)。
        ''' </summary>
        ''' <param name="sender">触发事件的实例。</param>
        ''' <param name="e">描述导致事件的条件的事件数据。</param>
        Private Sub CoreDispatcher_AcceleratorKeyActivated(sender As Windows.UI.Core.CoreDispatcher,
                                                           e As Windows.UI.Core.AcceleratorKeyEventArgs)
            Dim virtualKey As Windows.System.VirtualKey = e.VirtualKey

            ' 仅当按向左、向右或专用上一页或下一页键时才进一步
            ' 调查
            If (e.EventType = Windows.UI.Core.CoreAcceleratorKeyEventType.SystemKeyDown OrElse
                e.EventType = Windows.UI.Core.CoreAcceleratorKeyEventType.KeyDown) AndAlso
                (virtualKey = Windows.System.VirtualKey.Left OrElse
                virtualKey = Windows.System.VirtualKey.Right OrElse
                virtualKey = 166 OrElse
                virtualKey = 167) Then

                ' 确定按下了哪些修改键
                Dim coreWindow As Windows.UI.Core.CoreWindow = Window.Current.CoreWindow
                Dim downState As Windows.UI.Core.CoreVirtualKeyStates = Windows.UI.Core.CoreVirtualKeyStates.Down
                Dim menuKey As Boolean = (coreWindow.GetKeyState(Windows.System.VirtualKey.Menu) And downState) = downState
                Dim controlKey As Boolean = (coreWindow.GetKeyState(Windows.System.VirtualKey.Control) And downState) = downState
                Dim shiftKey As Boolean = (coreWindow.GetKeyState(Windows.System.VirtualKey.Shift) And downState) = downState
                Dim noModifiers As Boolean = Not menuKey AndAlso Not controlKey AndAlso Not shiftKey
                Dim onlyAlt As Boolean = menuKey AndAlso Not controlKey AndAlso Not shiftKey

                If (virtualKey = 166 AndAlso noModifiers) OrElse
                    (virtualKey = Windows.System.VirtualKey.Left AndAlso onlyAlt) Then

                    ' 在按上一页键或 Alt+向左键时向后导航
                    e.Handled = True
                    Me.GoBackCommand.Execute(Nothing)
                ElseIf (virtualKey = 167 AndAlso noModifiers) OrElse
                    (virtualKey = Windows.System.VirtualKey.Right AndAlso onlyAlt) Then

                    ' 在按下一页键或 Alt+向右键时向前导航
                    e.Handled = True
                    Me.GoForwardCommand.Execute(Nothing)
                End If
            End If
        End Sub

        ''' <summary>
        ''' 当此页处于活动状态并占用整个窗口时，在每次鼠标单击、触摸屏点击
        ''' 或执行等效交互时调用。    用于检测浏览器样式下一页和
        ''' 上一步鼠标按钮单击以在页之间导航。
        ''' </summary>
        ''' <param name="sender">触发事件的实例。</param>
        ''' <param name="e">描述导致事件的条件的事件数据。</param>
        Private Sub CoreWindow_PointerPressed(sender As Windows.UI.Core.CoreWindow,
                                              e As Windows.UI.Core.PointerEventArgs)
            Dim properties As Windows.UI.Input.PointerPointProperties = e.CurrentPoint.Properties

            ' 忽略与鼠标左键、右键和中键的键关联
            If properties.IsLeftButtonPressed OrElse properties.IsRightButtonPressed OrElse
                properties.IsMiddleButtonPressed Then Return

            ' 如果按下后退或前进(但不是同时)，则进行相应导航
            Dim backPressed As Boolean = properties.IsXButton1Pressed
            Dim forwardPressed As Boolean = properties.IsXButton2Pressed
            If backPressed Xor forwardPressed Then
                e.Handled = True
                If backPressed Then Me.GoBackCommand.Execute(Nothing)
                If forwardPressed Then Me.GoForwardCommand.Execute(Nothing)
            End If
        End Sub
#End If

#End Region

#Region "进程生命期管理"

        Private _pageKey As String

        ''' <summary>
        ''' 在此页将要在 Frame 中显示时进行调用。
        ''' </summary>
        ''' <param name="e">描述如何访问此页的事件数据。    Parameter
        ''' 属性提供要显示的组。</param>
        Public Sub OnNavigatedTo(e As Navigation.NavigationEventArgs)
            Dim frameState As Dictionary(Of String, Object) = SuspensionManager.SessionStateForFrame(Me.Frame)
            Me._pageKey = "Page-" & Me.Frame.BackStackDepth

            If e.NavigationMode = Navigation.NavigationMode.New Then

                ' 在向导航堆栈添加新页时清除向前导航的
                ' 现有状态
                Dim nextPageKey As String = Me._pageKey
                Dim nextPageIndex As Integer = Me.Frame.BackStackDepth
                While (frameState.Remove(nextPageKey))
                    nextPageIndex += 1
                    nextPageKey = "Page-" & nextPageIndex
                End While


                ' 将导航参数传递给新页
                RaiseEvent LoadState(Me, New LoadStateEventArgs(e.Parameter, Nothing))
            Else

                ' 通过将相同策略用于加载挂起状态并从缓存重新创建
                ' 放弃的页，将导航参数和保留页状态传递
                ' 给页
                RaiseEvent LoadState(Me, New LoadStateEventArgs(e.Parameter, DirectCast(frameState(Me._pageKey), Dictionary(Of String, Object))))
            End If
        End Sub

        ''' <summary>
        ''' 当此页不再在 Frame 中显示时调用。
        ''' </summary>
        ''' <param name="e">描述如何访问此页的事件数据。    Parameter
        ''' 属性提供要显示的组。</param>
        Public Sub OnNavigatedFrom(e As Navigation.NavigationEventArgs)
            Dim frameState As Dictionary(Of String, Object) = SuspensionManager.SessionStateForFrame(Me.Frame)
            Dim pageState As New Dictionary(Of String, Object)()
            RaiseEvent SaveState(Me, New SaveStateEventArgs(pageState))
            frameState(_pageKey) = pageState
        End Sub

        Public Event LoadState As LoadStateEventHandler
        Public Event SaveState As SaveStateEventHandler
#End Region

    End Class

    Public Class LoadStateEventArgs
        Inherits EventArgs

        Public Property NavigationParameter() As Object
        Public Property PageState() As Dictionary(Of String, Object)

        Public Sub New(navigationParameter As Object, pageState As Dictionary(Of String, Object))
            MyBase.New()
            Me.NavigationParameter = navigationParameter
            Me.PageState = pageState
        End Sub
    End Class
    Public Delegate Sub LoadStateEventHandler(sender As Object, e As LoadStateEventArgs)

    Public Class SaveStateEventArgs
        Inherits EventArgs

        Public Property PageState() As Dictionary(Of String, Object)

        Public Sub New(pageState As Dictionary(Of String, Object))
            MyBase.New()
            Me.PageState = pageState
        End Sub

    End Class

    Public Delegate Sub SaveStateEventHandler(sender As Object, e As SaveStateEventArgs)
End Namespace